/*
 Short description:
 ------------------
 Recently durring one of our code camps with Vesko Kolev we had the need of 
 testing an application which was time, CPU and memory consuming in different 
 multitasking environments. While Vesko has been codding the real stuff I did 
 the testing program given here. The idea of this program is to help you 
 understand how you can get most of your hardware. This is done by running 
 user selected program in selected count of threads and each thread has a 
 selected count of queued executions This can help you answer questions 
 like: "How much can this hardware take?", "Which is the optimal number 
 of parallel running processes for this hardware?" and so on. Since the 
 selected process is expected to run for a certain time and stop, after all 
 the processes are finished, the total time is displayed. Keep in mind that 
 I'm not using multimedia timers here and the precision (even displayed in 
 miliseconds) is limited by the system clock which ticks 17.2 times per 
 second. If you need better precision please fix the code and send your 
 fixes back to me :)

 Syntax: RuntimeTester [-p count] [-q count] cmd line
     -p : specifies the count of parallel running execution threads
     -q : specifies the count of queued runs per thread
	
 Examples:
 ---------
 RuntimeTester -p 2 -q 1 program.exe
 Will run 2 parallel queues and therefore program.exe will be started 2 times in parallel
 RuntimeTester will quit when both running copies stop executing

 RuntimeTester -p 1 -q 2 program.exe
 Will run 1 parallel queue and inside it program.exe will be started 2 times one after another
 RuntimeTester will quit when the last running copy of program.exe stop executing

 RuntimeTester -p 2 -q 3 program.exe
 Will run 2 parallel queues and inside it program.exe will be started 3 times one after another
 most of the time you will have 2 running copies of program.exe at the same time
 RuntimeTester will quit when all running copies stop executing
 the total number of executions of program.exe will be 6 times

 What's in archive:
 ------------------
 The source code and build batch files for 4 different compilers in both ANSI and UNICODE flavors: 
 MinGW, Microsoft Visual C/C++, OpenWatcom, DigitalMars C/C++. All the batch files include my 
 current path inside! You need to change them to your path (or remove the path if you have the 
 compiler(s) in your PATH environment variable).
*/

#include <stdio.h>
#include <tchar.h>
#include <windows.h>

#define MAX_CMD_LINE	4096

typedef struct {
	const TCHAR* cmdLine;
	int queueExecs;
} TThreadParam;

void showHelp(void)
{
	_putts(
_T("Syntax: RuntimeTester [-p count] [-q count] cmd line\n")
_T("    -p : specifies the count of parallel running execution threads\n")
_T("    -q : specifies the count of queued runs per thread\n")
);
	exit(1);
}

void printTime(const TCHAR* message)
{
	SYSTEMTIME st;
	GetSystemTime(&st);
	_tprintf(_T("[%02u.%02u.%04u %02u:%02u:%02u] "), st.wDay, st.wMonth, st.wYear, st.wHour, st.wMinute, st.wSecond);
	_putts(message);
	//_putts(_T("\n"));
}

int getNumber(const TCHAR* number)
{
	int res = _ttoi(number);
	if (!res) 
		showHelp();
	return res;
}

int checkParam(const TCHAR* param, int argc, TCHAR* argv[], int index)
{
	if ((index + 1 < argc) && (0 == _tcscmp(param, argv[index]))) {
		return getNumber(argv[index + 1]);
	}
	return 0;
}

DWORD WINAPI threadProc(LPVOID lpParameter)
{
	TThreadParam* pThreadParam = (TThreadParam *)lpParameter;
	
	STARTUPINFO si;
	ZeroMemory(&si, sizeof(STARTUPINFO));
	PROCESS_INFORMATION pi;
	ZeroMemory(&pi, sizeof(PROCESS_INFORMATION));

	for (int i = 0; i < pThreadParam->queueExecs; i++) {
		TCHAR* cmdLine = _tcsdup(pThreadParam->cmdLine);
		if (!CreateProcess(NULL, cmdLine, NULL, NULL, FALSE, 0, NULL, NULL, &si, &pi)) {
			_putts(_T("Error running selected process!"));
			exit(2);
		}
		WaitForSingleObject(pi.hProcess, INFINITE);
		CloseHandle(pi.hProcess);
		CloseHandle(pi.hThread);
		ZeroMemory(&pi, sizeof(PROCESS_INFORMATION));
		free(cmdLine);
	}

	return 0;
}

void createAndWaitThreads(int parallelExecs, int queueExecs, const TCHAR* cmdLine)
{
	TThreadParam threadParam;
	threadParam.cmdLine = cmdLine;
	threadParam.queueExecs = queueExecs;

	HANDLE* threadHandles = new HANDLE[parallelExecs];

	for (int i = 0; i < parallelExecs; i++) {
		threadHandles[i] = CreateThread(NULL, 0, threadProc, &threadParam, 0, NULL);
	}
	
	WaitForMultipleObjects((DWORD)parallelExecs, threadHandles, TRUE, INFINITE);

	delete threadHandles;
}

#ifdef __GNUC__
int main(int argc, char **a_argv)
#else // !__GNUC__
int _tmain(int argc, TCHAR **argv)
#endif // __GNUC__

{
	int parallelExecs = 1;
	int queueExecs = 1;
	int firstCommand = 3;
	DWORD startTicks, endTicks;
	TCHAR cmdLine[MAX_CMD_LINE] = _T("");

#ifdef __GNUC__
	TCHAR **argv;
	#ifdef UNICODE
		// MinGW doesn't support wmain() directly (shame on its CRT!)
		int a_argc;
		argv = CommandLineToArgvW(GetCommandLineW(), &a_argc);
	#else // !UNICODE
		argv = a_argv;
	#endif // UNICODE
#endif // __GNUC__

	_putts(_T("Runtime tester v1.0 - Copyright 2008 Demosten\nstjordanov@hotmail.com"));

	if (1 >= argc)
		showHelp();

	// get the params
	int temp;
	if (temp = checkParam(_T("-p"), argc, argv, 1))
		parallelExecs = temp;
	else if (temp = checkParam(_T("-q"), argc, argv, 1))
		queueExecs = temp;
	else
		firstCommand = 1;

	if (temp && (3 < argc)) {
		firstCommand = 5;
		if (temp = checkParam(_T("-p"), argc, argv, 3))
			parallelExecs = temp;
		else if (temp = checkParam(_T("-q"), argc, argv, 3))
			queueExecs = temp;
		else
			firstCommand = 3;
	}

	// prepare the command line
	for (; firstCommand < argc; firstCommand++) {
		_tcscat(cmdLine, argv[firstCommand]);
		_tcscat(cmdLine, _T(" "));
	}

	_putts(_T("Result command line:"));
	_putts(cmdLine);
	_putts(_T("\n"));

	printTime(_T("Start time"));
	startTicks = GetTickCount();

	createAndWaitThreads(parallelExecs, queueExecs, cmdLine);

	endTicks = GetTickCount();
	printTime(_T("End time"));
	_tprintf(_T("Total miliseconds: %u\n"), endTicks - startTicks);

	return 0;
}

